-- Modern Ambient Lighting Control
-- Redesigned UI/UX by Sayrx

local selectedColor = rgb(255, 100, 50)  -- Default warm orange
local activeZone = "headlights"  -- Default selection

-- Car paint colors
local paintColor = rgb(255, 255, 255)  -- Default white
local chameleonColorA = rgb(2, 0, 108)  -- Default chameleon A
local chameleonOpacityA = 0.5
local chameleonColorB = rgb(255, 77, 4)  -- Default chameleon B
local chameleonOpacityB = 0.8
local hasChameleon = false
local skinConfigPath = ""

-- Camera preview setup
local cameraCanvas = ui.ExtraCanvas(256, 192)
local cameraDummy = nil  -- Will look for CAMERA_HEADLIGHTS dummy

-- Light zone definitions
local lightZones = {
    headlights = {
        name = "Headlights",
        meshes = {"light1_0", "light1_1", "light1_2", "light1_3", "light1_4", "light1_5", 
                  "light1_6", "light1_7", "light1_8", "light1_9", "light1_10", 
                  "light1_11", "light1_12", "light1_13", "light1_14", "light1_15", 
                  "light1_16", "light1_17", "light1_18", "light1_19"},
        available = nil  -- Will be checked on first run
    },
    ambient = {
        name = "Ambient",
        meshes = {"light2_0", "light2_1", "light2_2", "light2_3", "light2_4", "light2_5", 
                  "light2_6", "light2_7", "light2_8", "light2_9", "light2_10", "light2_11", 
                  "light2_12", "light2_13", "light2_14", "light2_15", "light2_16", "light2_17", 
                  "light2_18", "light2_19"},
        available = nil  -- Will be checked on first run
    },
    carpaint = {
        name = "Car Paint",
        meshes = {},  -- Will look for CarPaint material
        available = true  -- Always available
    }
}

-- Check if any meshes exist for a zone
local function checkZoneAvailability(zone)
    for _, meshName in ipairs(zone.meshes) do
        local mesh = ac.findMeshes(meshName)
        if mesh and mesh:size() > 0 then
            return true  -- At least one mesh exists
        end
    end
    return false  -- No meshes found
end

-- Load chameleon settings from skin ext_config.ini
local function loadChameleonSettings()
    -- Get current skin path using correct AC API
    local carIndex = 0  -- Player car
    skinConfigPath = ac.getFolder(ac.FolderID.ContentCars) .. "/" .. ac.getCarID(carIndex) .. "/skins/" .. ac.getCarSkinID(carIndex) .. "/ext_config.ini"
    
    local file = io.open(skinConfigPath, 'r')
    if not file then 
        hasChameleon = false
        return 
    end
    
    local content = file:read('*all')
    file:close()
    
    -- Check if chameleon section exists
    hasChameleon = string.find(content, "%[Material_CarPaint_Chameleon%]") ~= nil
    
    if hasChameleon then
        -- Parse ChameleonColorA
        local colorA, opacityA = string.match(content, "ChameleonColorA%s*=%s*#(%x+)%s*,%s*([%d%.]+)")
        if colorA and opacityA then
            local r = tonumber(string.sub(colorA, 1, 2), 16) or 2
            local g = tonumber(string.sub(colorA, 3, 4), 16) or 0
            local b = tonumber(string.sub(colorA, 5, 6), 16) or 108
            chameleonColorA = rgb(r, g, b)
            chameleonOpacityA = tonumber(opacityA) or 0.5
        end
        
        -- Parse ChameleonColorB
        local colorB, opacityB = string.match(content, "ChameleonColorB%s*=%s*#(%x+)%s*,%s*([%d%.]+)")
        if colorB and opacityB then
            local r = tonumber(string.sub(colorB, 1, 2), 16) or 255
            local g = tonumber(string.sub(colorB, 3, 4), 16) or 77
            local b = tonumber(string.sub(colorB, 5, 6), 16) or 4
            chameleonColorB = rgb(r, g, b)
            chameleonOpacityB = tonumber(opacityB) or 0.8
        end
    end
end

-- Save chameleon settings to skin ext_config.ini and apply to car
local function saveChameleonSettings()
    if not hasChameleon or skinConfigPath == "" then return end
    
    local file = io.open(skinConfigPath, 'r')
    if not file then return end
    
    local content = file:read('*all')
    file:close()
    
    -- Convert RGB to hex
    local function rgbToHex(color)
        return string.format("%02x%02x%02x", 
            math.floor(color.r * 255), 
            math.floor(color.g * 255), 
            math.floor(color.b * 255))
    end
    
    -- Update ChameleonColorA
    content = string.gsub(content, 
        "(ChameleonColorA%s*=%s*)#%x+%s*,%s*[%d%.]+",
        "%1#" .. rgbToHex(chameleonColorA) .. ", " .. string.format("%.1f", chameleonOpacityA))
    
    -- Update ChameleonColorB
    content = string.gsub(content, 
        "(ChameleonColorB%s*=%s*)#%x+%s*,%s*[%d%.]+",
        "%1#" .. rgbToHex(chameleonColorB) .. ", " .. string.format("%.1f", chameleonOpacityB))
    
    file = io.open(skinConfigPath, 'w')
    if file then
        file:write(content)
        file:close()
    end
    
    -- Apply chameleon colors to car material properties in real-time
    local carMeshes = ac.findMeshes('shader:smCarPaint_chameleon')
    if carMeshes and carMeshes:size() > 0 then
        -- Set chameleon material properties
        carMeshes:setMaterialProperty('ksAmbient', chameleonColorA:rgbm(chameleonOpacityA))
        carMeshes:setMaterialProperty('ksDiffuse', chameleonColorB:rgbm(chameleonOpacityB))
    end
end

-- Apply car paint color to metal_detail.dds
local function applyCarPaint(color)
    paintColor = color
    -- Find meshes with "smCarPaint_chameleon" shader
    local carMeshes = ac.findMeshes('shader:smCarPaint_chameleon')
    if carMeshes and carMeshes:size() > 0 then
        carMeshes:setMaterialTexture('txDetail', color:rgbm(1))
    end
end

-- Initialize zone availability on first run
local initialized = false
local function initializeZones()
    if initialized then return end
    
    for zoneName, zone in pairs(lightZones) do
        if zoneName ~= "carpaint" then
            zone.available = checkZoneAvailability(zone)
        end
    end
    
    -- Load chameleon settings from current skin
    loadChameleonSettings()
    
    -- Set default active zone to first available one
    if not lightZones.headlights.available and lightZones.ambient.available then
        activeZone = "ambient"
    elseif lightZones.headlights.available then
        activeZone = "headlights"
    end
    
    initialized = true
end

-- Preset colors for quick selection
local presetColors = {
    {name = "Red", r = 255, g = 0, b = 0},
    {name = "Orange", r = 255, g = 100, b = 0},
    {name = "Yellow", r = 255, g = 200, b = 0},
    {name = "Green", r = 0, g = 255, b = 50},
    {name = "Cyan", r = 0, g = 200, b = 255},
    {name = "Blue", r = 0, g = 100, b = 255},
    {name = "Purple", r = 150, g = 0, b = 255},
    {name = "Pink", r = 255, g = 0, b = 150},
    {name = "White", r = 255, g = 255, b = 255}
}

-- Apply color to selected zone
local function applyColor(color)
    if activeZone == "carpaint" then
        applyCarPaint(color)
        return
    end
    
    local zone = lightZones[activeZone]
    if not zone or not zone.available then return end
    
    for _, meshName in ipairs(zone.meshes) do
        ac.findMeshes(meshName):setMaterialTexture('txDiffuse', color:rgbm(1))
    end
end

return function(dt)
    -- Initialize zones on first run
    initializeZones()
    
    -- Find camera dummy on first run
    if not cameraDummy then
        cameraDummy = ac.findNodes('CAMERA_HEADLIGHTS')
        if not cameraDummy or cameraDummy:empty() then
            cameraDummy = false  -- Mark as not found to avoid repeated searches
        end
    end
    
    -- Render camera preview if dummy exists
    if cameraDummy and lightZones.headlights.available then
        cameraCanvas:update(function()
            local carNode = ac.findNodes('carRoot:0')
            if not carNode then return end
            
            local carMeshes = carNode:findMeshes('{ lod:A }')
            local geometryShot = ac.GeometryShot(carMeshes, vec2(256, 192), 1, false, render.AntialiasingMode.None)
            
            if geometryShot then
                -- Get dummy world transformation
                local dummyTransform = cameraDummy:getWorldTransformationRaw()
                
                -- Set clipping planes (near, far) - reduce near clip to avoid cutting off close objects
                geometryShot:setClippingPlanes(0.01, 600)  -- Near: 1cm, Far: 100m
                
                -- Update geometry shot with camera from dummy position
                geometryShot:update(
                    dummyTransform.position,
                    dummyTransform.look,
                    dummyTransform.up,
                    8  -- FOV in degrees (ultra-wide view to show whole scene)
                )
                
                -- Draw the shot
                ui.drawImage(geometryShot, vec2(0, 0), vec2(256, 192))
                
                -- Clean up
                geometryShot:dispose()
            end
        end)
    end
    
    -- Modern header bar
    ui.drawRectFilled(vec2(0, 0), vec2(1080, 70), rgbm(0.08, 0.08, 0.1, 1))
    ui.drawRectFilled(vec2(0, 68), vec2(1080, 72), rgbm(0.4, 0.2, 0.7, 1))  -- Purple accent line
    
    display.text{
        text = "AMBIENT LIGHTING",
        pos = vec2(540, 22),
        letter = vec2(20, 30),
        font = "c7_new",
        alignment = 0,
        spacing = -3,
        color = rgbm(1, 1, 1, 1)
    }
    
    display.text{
        text = "CONTROL SYSTEM",
        pos = vec2(540, 48),
        letter = vec2(12, 18),
        font = "c7_new",
        alignment = 0,
        spacing = -2,
        color = rgbm(0.6, 0.6, 0.6, 1)
    }
    
    -- Zone selection buttons with better styling (3 buttons now)
    local buttonWidth = 180
    local buttonHeight = 55
    local buttonY = 90
    local spacing = 25
    local startX = (1080 - (buttonWidth * 3 + spacing * 2)) / 2
    
    -- Headlights button
    local headlightsX = startX
    local headlightsActive = activeZone == "headlights"
    local headlightsAvailable = lightZones.headlights.available
    
    -- Button background (grayed out if disabled)
    local headlightsBgColor
    if not headlightsAvailable then
        headlightsBgColor = rgbm(0.1, 0.1, 0.12, 0.5)  -- Disabled gray
    elseif headlightsActive then
        headlightsBgColor = rgbm(0.45, 0.25, 0.75, 1)  -- Active purple
    else
        headlightsBgColor = rgbm(0.15, 0.15, 0.17, 1)  -- Inactive dark
    end
    
    ui.drawRectFilled(
        vec2(headlightsX, buttonY), 
        vec2(headlightsX + buttonWidth, buttonY + buttonHeight),
        headlightsBgColor, 
        12
    )
    
    -- Button border/outline
    ui.drawRect(
        vec2(headlightsX, buttonY), 
        vec2(headlightsX + buttonWidth, buttonY + buttonHeight),
        headlightsAvailable and (headlightsActive and rgbm(0.6, 0.4, 0.9, 1) or rgbm(0.3, 0.3, 0.35, 1)) or rgbm(0.2, 0.2, 0.22, 1), 
        12, 
        2
    )
    
    -- Only allow hover/click if available
    if headlightsAvailable and ui.rectHovered(vec2(headlightsX, buttonY), vec2(headlightsX + buttonWidth, buttonY + buttonHeight)) then
        ui.drawRectFilled(
            vec2(headlightsX, buttonY), 
            vec2(headlightsX + buttonWidth, buttonY + buttonHeight),
            rgbm(0.5, 0.3, 0.8, 0.4), 
            12
        )
        if ui.mouseClicked() then
            activeZone = "headlights"
        end
    end
    
    display.text{
        text = "HEADLIGHTS",
        pos = vec2(headlightsX + 20, buttonY + 27),
        letter = vec2(16, 24),
        font = "c7_new",
        alignment = -1.5,
        spacing = -2,
        color = headlightsAvailable and (headlightsActive and rgbm(1, 1, 1, 1) or rgbm(0.65, 0.65, 0.65, 1)) or rgbm(0.3, 0.3, 0.3, 0.7)
    }
    
    -- Ambient button
    local ambientX = headlightsX + buttonWidth + spacing
    local ambientActive = activeZone == "ambient"
    local ambientAvailable = lightZones.ambient.available
    
    -- Button background (grayed out if disabled)
    local ambientBgColor
    if not ambientAvailable then
        ambientBgColor = rgbm(0.1, 0.1, 0.12, 0.5)  -- Disabled gray
    elseif ambientActive then
        ambientBgColor = rgbm(0.45, 0.25, 0.75, 1)  -- Active purple
    else
        ambientBgColor = rgbm(0.15, 0.15, 0.17, 1)  -- Inactive dark
    end
    
    ui.drawRectFilled(
        vec2(ambientX, buttonY), 
        vec2(ambientX + buttonWidth, buttonY + buttonHeight),
        ambientBgColor, 
        12
    )
    
    -- Button border/outline
    ui.drawRect(
        vec2(ambientX, buttonY), 
        vec2(ambientX + buttonWidth, buttonY + buttonHeight),
        ambientAvailable and (ambientActive and rgbm(0.6, 0.4, 0.9, 1) or rgbm(0.3, 0.3, 0.35, 1)) or rgbm(0.2, 0.2, 0.22, 1), 
        12, 
        2
    )
    
    -- Only allow hover/click if available
    if ambientAvailable and ui.rectHovered(vec2(ambientX, buttonY), vec2(ambientX + buttonWidth, buttonY + buttonHeight)) then
        ui.drawRectFilled(
            vec2(ambientX, buttonY), 
            vec2(ambientX + buttonWidth, buttonY + buttonHeight),
            rgbm(0.5, 0.3, 0.8, 0.4), 
            12
        )
        if ui.mouseClicked() then
            activeZone = "ambient"
        end
    end
    
    local ambientTextColor = ambientAvailable and (ambientActive and rgbm(1, 1, 1, 1) or rgbm(0.65, 0.65, 0.65, 1)) or rgbm(0.3, 0.3, 0.3, 0.7)
    display.text{
        text = "AMBIENT",
        pos = vec2(ambientX + 20, buttonY + 27),
        letter = vec2(16, 24),
        font = "c7_new",
        alignment = -1.5,
        spacing = -2,
        color = ambientTextColor
    }
    
    -- Car Paint button (new third button)
    local carpaintX = ambientX + buttonWidth + spacing
    local carpaintActive = activeZone == "carpaint"
    local carpaintAvailable = lightZones.carpaint.available
    
    -- Button background
    local carpaintBgColor
    if carpaintActive then
        carpaintBgColor = rgbm(0.45, 0.25, 0.75, 1)  -- Active purple
    else
        carpaintBgColor = rgbm(0.15, 0.15, 0.17, 1)  -- Inactive dark
    end
    
    ui.drawRectFilled(
        vec2(carpaintX, buttonY), 
        vec2(carpaintX + buttonWidth, buttonY + buttonHeight),
        carpaintBgColor, 
        12
    )
    
    -- Button border/outline
    ui.drawRect(
        vec2(carpaintX, buttonY), 
        vec2(carpaintX + buttonWidth, buttonY + buttonHeight),
        carpaintActive and rgbm(0.6, 0.4, 0.9, 1) or rgbm(0.3, 0.3, 0.35, 1), 
        12, 
        2
    )
    
    -- Hover effect and click handler
    if ui.rectHovered(vec2(carpaintX, buttonY), vec2(carpaintX + buttonWidth, buttonY + buttonHeight)) then
        ui.drawRectFilled(
            vec2(carpaintX, buttonY), 
            vec2(carpaintX + buttonWidth, buttonY + buttonHeight),
            rgbm(0.5, 0.3, 0.8, 0.4), 
            12
        )
        if ui.mouseClicked() then
            activeZone = "carpaint"
        end
    end
    
    display.text{
        text = "CAR PAINT",
        pos = vec2(carpaintX + 20, buttonY + 27),
        letter = vec2(16, 24),
        font = "c7_new",
        alignment = -1.5,
        spacing = -2,
        color = carpaintActive and rgbm(1, 1, 1, 1) or rgbm(0.65, 0.65, 0.65, 1)
    }
    
    -- Color picker section with better frame
    ui.setCursor(vec2(35, 170))
    
    -- Show different UI based on active zone
    if activeZone == "carpaint" and hasChameleon then
        -- Chameleon paint mode - show 2 color pickers for chameleon colors
        ui.drawRectFilled(vec2(35, 170), vec2(1045, 430), rgbm(0.12, 0.12, 0.14, 1), 10)
        ui.drawRect(vec2(35, 170), vec2(1045, 430), rgbm(0.25, 0.25, 0.28, 1), 10, 2)
        
        display.text{
            text = "CHAMELEON PAINT COLORS",
            pos = vec2(50, 180),
            letter = vec2(12, 18),
            font = "c7_new",
            alignment = -1.5,
            spacing = -2,
            color = rgbm(0.8, 0.8, 0.8, 1)
        }
        
        -- Chameleon Color A
        display.text{
            text = "COLOR A",
            pos = vec2(50, 210),
            letter = vec2(10, 15),
            font = "c7_new",
            alignment = -1.5,
            spacing = -2,
            color = rgbm(0.7, 0.7, 0.7, 1)
        }
        
        ui.setCursor(vec2(40, 230))
        ui.childWindow('ChameleonA', vec2(480, 185), function()
            if ui.colorPicker("##ColorA", chameleonColorA, 
                ui.ColorPickerFlags.PickerHueWheel + 
                ui.ColorPickerFlags.NoSidePreview + 
                ui.ColorPickerFlags.NoLabel + 
                ui.ColorPickerFlags.NoAlpha) then
                saveChameleonSettings()
            end
            
            ui.text("Opacity:")
            ui.sameLine()
            if ui.slider("##OpacityA", chameleonOpacityA, 0, 1, "%.2f") then
                saveChameleonSettings()
            end
        end)
        
        -- Chameleon Color B
        display.text{
            text = "COLOR B",
            pos = vec2(540, 210),
            letter = vec2(10, 15),
            font = "c7_new",
            alignment = -1.5,
            spacing = -2,
            color = rgbm(0.7, 0.7, 0.7, 1)
        }
        
        ui.setCursor(vec2(530, 230))
        ui.childWindow('ChameleonB', vec2(480, 185), function()
            if ui.colorPicker("##ColorB", chameleonColorB, 
                ui.ColorPickerFlags.PickerHueWheel + 
                ui.ColorPickerFlags.NoSidePreview + 
                ui.ColorPickerFlags.NoLabel + 
                ui.ColorPickerFlags.NoAlpha) then
                saveChameleonSettings()
            end
            
            ui.text("Opacity:")
            ui.sameLine()
            if ui.slider("##OpacityB", chameleonOpacityB, 0, 1, "%.2f") then
                saveChameleonSettings()
            end
        end)
        
    elseif activeZone == "carpaint" then
        -- Regular car paint mode (no chameleon)
        ui.drawRectFilled(vec2(35, 170), vec2(495, 430), rgbm(0.12, 0.12, 0.14, 1), 10)
        ui.drawRect(vec2(35, 170), vec2(495, 430), rgbm(0.25, 0.25, 0.28, 1), 10, 2)
        
        display.text{
            text = "CAR PAINT COLOR",
            pos = vec2(50, 180),
            letter = vec2(12, 18),
            font = "c7_new",
            alignment = -1.5,
            spacing = -2,
            color = rgbm(0.8, 0.8, 0.8, 1)
        }
        
        ui.setCursor(vec2(40, 205))
        ui.childWindow('PaintPicker', vec2(400, 220), function()
            if ui.colorPicker("##PaintColor", paintColor, 
                ui.ColorPickerFlags.PickerHueWheel + 
                ui.ColorPickerFlags.NoSidePreview + 
                ui.ColorPickerFlags.NoLabel + 
                ui.ColorPickerFlags.NoAlpha) then
                applyCarPaint(paintColor)
            end
        end)
        
        -- Presets for car paint
        ui.setCursor(vec2(515, 170))
        ui.drawRectFilled(vec2(515, 170), vec2(1045, 430), rgbm(0.12, 0.12, 0.14, 1), 10)
        ui.drawRect(vec2(515, 170), vec2(1045, 430), rgbm(0.25, 0.25, 0.28, 1), 10, 2)
        
        display.text{
            text = "QUICK PRESETS",
            pos = vec2(530, 180),
            letter = vec2(12, 18),
            font = "c7_new",
            alignment = -1.5,
            spacing = -2,
            color = rgbm(0.8, 0.8, 0.8, 1)
        }
        
        ui.setCursor(vec2(520, 205))
        ui.childWindow('PaintPresets', vec2(520, 220), function()
            local presetWidth = 160
            local presetHeight = 45
            local presetSpacing = 10
            local columns = 3
            
            for i, preset in ipairs(presetColors) do
                local col = (i - 1) % columns
                
                if col > 0 then ui.sameLine(0, presetSpacing) end
                
                local presetColor = rgb(preset.r, preset.g, preset.b)
                ui.pushStyleColor(ui.StyleColor.Button, presetColor:rgbm(1))
                ui.pushStyleColor(ui.StyleColor.ButtonHovered, presetColor:rgbm(1.3))
                ui.pushStyleColor(ui.StyleColor.ButtonActive, presetColor:rgbm(0.7))
                ui.pushStyleVar(ui.StyleVar.FrameRounding, 8)
                
                if ui.button(preset.name.."##paint"..i, vec2(presetWidth, presetHeight)) then
                    paintColor = rgb(preset.r, preset.g, preset.b)
                    applyCarPaint(paintColor)
                end
                
                ui.popStyleVar()
                ui.popStyleColor(3)
            end
        end)
        
    else
        -- Lighting mode (headlights/ambient)
        ui.drawRectFilled(vec2(35, 170), vec2(495, 430), rgbm(0.12, 0.12, 0.14, 1), 10)
        ui.drawRect(vec2(35, 170), vec2(495, 430), rgbm(0.25, 0.25, 0.28, 1), 10, 2)
        
        display.text{
            text = "CUSTOM COLOR",
            pos = vec2(50, 180),
            letter = vec2(12, 18),
            font = "c7_new",
            alignment = -1.5,
            spacing = -2,
            color = rgbm(0.8, 0.8, 0.8, 1)
        }
        
        ui.setCursor(vec2(40, 205))
        ui.childWindow('ColorPicker', vec2(400, 220), function()
            if ui.colorPicker("##Color", selectedColor, 
                ui.ColorPickerFlags.PickerHueWheel + 
                ui.ColorPickerFlags.NoSidePreview + 
                ui.ColorPickerFlags.NoLabel + 
                ui.ColorPickerFlags.NoAlpha) then
                applyColor(selectedColor)
            end
        end)
        
        -- Preset colors section
        ui.setCursor(vec2(515, 170))
        ui.drawRectFilled(vec2(515, 170), vec2(1045, 430), rgbm(0.12, 0.12, 0.14, 1), 10)
        ui.drawRect(vec2(515, 170), vec2(1045, 430), rgbm(0.25, 0.25, 0.28, 1), 10, 2)
        
        display.text{
            text = "QUICK PRESETS",
            pos = vec2(530, 180),
            letter = vec2(12, 18),
            font = "c7_new",
            alignment = -1.5,
            spacing = -2,
            color = rgbm(0.8, 0.8, 0.8, 1)
        }
        
        ui.setCursor(vec2(520, 205))
        ui.childWindow('Presets', vec2(520, 220), function()
            local presetWidth = 160
            local presetHeight = 45
            local presetSpacing = 10
            local columns = 3
            
            for i, preset in ipairs(presetColors) do
                local col = (i - 1) % columns
                
                if col > 0 then ui.sameLine(0, presetSpacing) end
                
                local presetColor = rgb(preset.r, preset.g, preset.b)
                ui.pushStyleColor(ui.StyleColor.Button, presetColor:rgbm(1))
                ui.pushStyleColor(ui.StyleColor.ButtonHovered, presetColor:rgbm(1.3))
                ui.pushStyleColor(ui.StyleColor.ButtonActive, presetColor:rgbm(0.7))
                ui.pushStyleVar(ui.StyleVar.FrameRounding, 8)
                
                if ui.button(preset.name.."##"..i, vec2(presetWidth, presetHeight)) then
                    selectedColor = rgb(preset.r, preset.g, preset.b)
                    applyColor(selectedColor)
                end
                
                ui.popStyleVar()
                ui.popStyleColor(3)
            end
        end)
    end
    
    -- Headlight camera preview (bottom right corner)
    if cameraDummy and lightZones.headlights.available then
        local previewWidth, previewHeight = 280, 210  -- Wider window (was 200x150)
        local previewX = 1080 - previewWidth - 35  -- Right edge minus width minus padding
        local previewY = 525
        
        -- Window background frame
        ui.drawRectFilled(
            vec2(previewX - 5, previewY - 25), 
            vec2(previewX + previewWidth + 5, previewY + previewHeight + 5), 
            rgbm(0.08, 0.08, 0.1, 1), 
            8
        )
        
        -- Window title
        display.text{
            text = "HEADLIGHT VIEW",
            pos = vec2(previewX + previewWidth/2, previewY - 15),
            letter = vec2(10, 15),
            font = "c7_new",
            alignment = 0,
            spacing = -1,
            color = rgbm(0.5, 0.25, 0.8, 1)
        }
        
        -- Camera feed with purple border
        ui.drawRect(
            vec2(previewX, previewY), 
            vec2(previewX + previewWidth, previewY + previewHeight), 
            rgbm(0.5, 0.25, 0.8, 1), 
            4, 
            2
        )
        ui.drawImage(
            cameraCanvas, 
            vec2(previewX, previewY), 
            vec2(previewX + previewWidth, previewY + previewHeight)
        )
    end
    
    -- Professional footer with prominent branding
    ui.drawRectFilled(vec2(0, 445), vec2(1080, 480), rgbm(0.08, 0.08, 0.1, 1))
    ui.drawRectFilled(vec2(0, 445), vec2(1080, 448), rgbm(0.5, 0.25, 0.8, 1))  -- Bright purple accent line
    
    -- Glow effect behind text
    display.text{
        text = "MADE BY SAYRX",
        pos = vec2(541, 463),
        letter = vec2(18, 26),
        font = "c7_new",
        alignment = 0,
        spacing = -1,
        color = rgbm(0.5, 0.25, 0.8, 0.5)  -- Purple glow
    }
    
    -- Main text
    display.text{
        text = "MADE BY SAYRX",
        pos = vec2(540, 462),
        letter = vec2(18, 26),
        font = "c7_new",
        alignment = 0,
        spacing = -1,
        color = rgbm(1, 1, 1, 1)
    }
end